'use strict';

/* --------------------------------------------------------------
 datepicker.js 2018-08-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Datepicker Widget
 *
 * Creates a customizable date(range)picker.
 *
 * jQueryUI Datepicker API: {@link http://api.jqueryui.com/datepicker}
 *
 * You can add the `data-datepicker-gx-container` attribute and it will style the datepicker with
 * the new CSS styles located at the gx-admin.css file. This might be useful when the .gx-container
 * class is not set directly on the <body> tag but in an inner div element of the page. The datepicker
 * will create a new div element which might be outside the .gx-container and therefore will not have
 * its style. This widget is already styled in Honeygrid.
 *
 * ### Example
 *
 * When the page loads, an input field as a date picker will be added.
 *
 * ```html
 * <input type="text" data-jse-widget="datepicker" data-datepicker-show-On="focus"
 *      data-datepicker-gx-container placeholder="##.##.####" />
 * ```
 *
 * For custom date format, use the 'data-datepicker-format' attribute.
 *
 * @module JSE/Widgets/datepicker
 * @requires jQueryUI-Library
 */
jse.widgets.module('datepicker', [jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.css', jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.js'],

/** @lends module:Widgets/datepicker */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Widget Reference
     *
     * @type {object}
     */
    $this = $(this),


    /**
     * Default Options for Widget
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Final Widget Options
     *
     * @type {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONALITY
    // ------------------------------------------------------------------------

    /**
     * Update Timestamp Field
     *
     * Function that updates the timestamp field belonging to this datepicker. If no
     * one exists, it gets generated.
     *
     * @param {object} inst jQuery datepicker instance object.
     */
    var _updateTimestampField = function _updateTimestampField(inst) {
        var name = $this.attr('name'),
            $ts = $this.siblings('[name="ts_' + name + '"]'),
            value = new Date([inst.selectedYear, inst.selectedMonth + 1, inst.selectedDay].join(', ')).valueOf();
        if (!$ts.length) {
            $this.after('<input type="hidden" name="ts_' + name + '" value="' + value + '"/>');
        } else {
            $ts.val(value);
        }
    };

    /**
     * Get Configuration
     *
     * Function to create the datepicker configuration object.
     *
     * @returns {object} JSON-configuration object.
     */
    var _getConfiguration = function _getConfiguration() {
        // Set default min / max values.
        options.max = options.max ? new Date(options.max) : null;
        options.min = options.min ? new Date(options.min) : null;

        // Base Configuration
        var configuration = {
            constrainInput: true,
            showOn: 'focus',
            showWeek: true,
            changeMonth: true,
            changeYear: true,
            minDate: options.min,
            maxDate: options.max,
            dayNamesMin: [jse.core.lang.translate('_SUNDAY_SHORT', 'general'), jse.core.lang.translate('_MONDAY_SHORT', 'general'), jse.core.lang.translate('_TUESDAY_SHORT', 'general'), jse.core.lang.translate('_WEDNESDAY_SHORT', 'general'), jse.core.lang.translate('_THURSDAY_SHORT', 'general'), jse.core.lang.translate('_FRIDAY_SHORT', 'general'), jse.core.lang.translate('_SATURDAY_SHORT', 'general')],
            dayNames: [jse.core.lang.translate('_SUNDAY', 'general'), jse.core.lang.translate('_MONDAY', 'general'), jse.core.lang.translate('_TUESDAY', 'general'), jse.core.lang.translate('_WEDNESDAY', 'general'), jse.core.lang.translate('_THURSDAY', 'general'), jse.core.lang.translate('_FRIDAY', 'general'), jse.core.lang.translate('_SATURDAY', 'general')],
            monthNamesShort: [jse.core.lang.translate('_JANUARY_SHORT', 'general'), jse.core.lang.translate('_FEBRUARY_SHORT', 'general'), jse.core.lang.translate('_MARCH_SHORT', 'general'), jse.core.lang.translate('_APRIL_SHORT', 'general'), jse.core.lang.translate('_MAY_SHORT', 'general'), jse.core.lang.translate('_JUNE_SHORT', 'general'), jse.core.lang.translate('_JULY_SHORT', 'general'), jse.core.lang.translate('_AUGUST_SHORT', 'general'), jse.core.lang.translate('_SEPTEMBER_SHORT', 'general'), jse.core.lang.translate('_OCTOBER_SHORT', 'general'), jse.core.lang.translate('_NOVEMBER_SHORT', 'general'), jse.core.lang.translate('_DECEMBER_SHORT', 'general')],
            monthNames: [jse.core.lang.translate('_JANUARY', 'general'), jse.core.lang.translate('_FEBRUARY', 'general'), jse.core.lang.translate('_MARCH', 'general'), jse.core.lang.translate('_APRIL', 'general'), jse.core.lang.translate('_MAY', 'general'), jse.core.lang.translate('_JUNE', 'general'), jse.core.lang.translate('_JULY', 'general'), jse.core.lang.translate('_AUGUST', 'general'), jse.core.lang.translate('_SEPTEMBER', 'general'), jse.core.lang.translate('_OCTOBER', 'general'), jse.core.lang.translate('_NOVEMBER', 'general'), jse.core.lang.translate('_DECEMBER', 'general')],
            prevText: jse.core.lang.translate('_PREV', 'general'),
            nextText: jse.core.lang.translate('_NEXT', 'general'),
            onSelect: function onSelect(date, inst) {
                _updateTimestampField(inst);
            }
        };

        // Set "showOn" options.
        if (options.showOn) {
            configuration.showOn = options.showOn;
        }

        // Sets the alternative field with an other date format (for backend).
        if (options.alt) {
            configuration.altField = options.alt;
            configuration.altFormat = '@';
        }

        // Trigger an event onSelect to inform dependencies and set the min / max value at the
        // current value of the dependency.
        if (options.depends && options.type) {
            var $depends = $(options.depends),
                value = $depends.val(),
                type = options.type === 'max' ? 'min' : 'max';

            // Add callback to the onSelect-Event.
            configuration.onSelect = function (date, inst) {
                _updateTimestampField(inst);
                var payload = {
                    'type': options.type,
                    'date': [inst.selectedYear, inst.selectedMonth + 1, inst.selectedDay].join(', ')
                };
                $depends.trigger('datepicker.selected', [payload]);
            };

            // Get and set the current value of the dependency.
            if (value) {
                var date = $.datepicker.parseDate($.datepicker._defaults.dateFormat, value);
                configuration[type + 'Date'] = date;
            }
        }

        // Override date format with data attribute value
        configuration.dateFormat = data.format || jse.core.config.get('languageCode') === 'de' ? 'dd.mm.yy' : 'mm/dd/yy';

        // Merge the data array with the datepicker array for enabling the original widget API options.
        configuration = $.extend(true, {}, configuration, data);

        return configuration;
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    /**
     * Initialize method of the widget, called by the engine.
     */
    module.init = function (done) {
        // Enable the datepicker widget.
        var configuration = _getConfiguration();
        $this.datepicker(configuration);

        // Get the gx-container style (newer style).
        if (typeof options.gxContainer !== 'undefined') {
            $(document).find('.ui-datepicker').not('.gx-container').addClass('gx-container');
        }

        // Add event listener for other datepickers to set the min / maxDate (for daterange).
        $this.on('datepicker.selected', function (e, d) {
            $this.datepicker('option', d.type + 'Date', new Date(d.date));
        });

        done();
    };

    // Return data to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
